define([
    'underscore',
    'jquery',
    'backbone',
    'App',
    'Global',
    'MainView',
    'modules/landing-page/landing-page-view',
    'modules/page/footer/footer-view',
    'User',
    'modules/connection/server-error-popup-view',
    'modules/right-of-access/right-of-access_module',
    'modules/appointments/appointments-module',
    'modules/notification-preferences/notification-preferences-module',
    'modules/notifications/notifications-module',
    'modules/vha-register/vha-register-view',
], function(
    _,
    $,
    Backbone,
    app,
    Global,
    MainView,
    LandingPageView,
    FooterView,
    User,
    ServerErrorPopupView,
    roaModule,
    appointmentsModule,
    notificationPreferencesModule,
    notificationsModule,
    VHARegisterView
) {
    'use strict';

    var utilities = Global.getUtilities();


    app.show = function (view) {
        var content = this.mainView.getRegion('content');
        content.show(view);
    };

    /**
     * Primary entry point to the application
     */
    app.on('start', function () {
        var roa = new $.Deferred();

        this.mainView = new MainView();
        this.mainView.render();

        prepareCoreComponents();
        prepareUser();
        prepareApiKeys();
        prepareRoa(roa);
        preparePatientIdentifiers(roa);
        prepareCcTypes(roa);
        prepareCcTypesFail();
        prepareFacilities(roa);
        prepareFacilitiesFail();
        prepareTracking(roa);
        prepareRequests(roa);
        prepareNotifications(roa);
        prepareNotificationPreferences(roa);
        preparePatientEnrolledFacilities(roa);
        prepareAppointments(roa);
        prepareCCEligibilityDetermination();

    });


    /**
     * Launches the Server Error Popup
     * @return {void}
     */
    function onError() {
        var error = new ServerErrorPopupView({
            elAfterClose: $('#header h1'),
        });
        error.openPopup();
    }


    /**
     * All prepare functions in this file consist of a list of dependencies for the object they are preparing
     * followed by what function telling it how to handle the success and or error.
     *
     * Even when a dependency is not a direct requirement of the resource, it should be included to avoid confusion
     * surrounded the oder of events.
     *
     * @return {void}
     */
    function prepareCoreComponents() {
        startSession();

        startApiMonitoring();
        startDisabledFeatures();
        getConfigurations();
    }


    /**
     * All prepare functions in this file consist of a list of dependencies for the object they are preparing
     * followed by what function telling it how to handle the success and or error.
     *
     * Even when a dependency is not a direct requirement of the resource, it should be included to avoid confusion
     * surrounded the oder of events.
     *
     * @return {void}
     */
    function prepareUser() {
        var when = $.when(
            app.userSession.isReady,
            app.apiKeys.isReady,
            app.disabledFeatures.isReady,
            app.configurations.isReady
        );

        when.then(startUser);
        when.fail(onError);
    }


    /**
     * All prepare functions in this file consist of a list of dependencies for the object they are preparing
     * followed by what function telling it how to handle the success and or error.
     *
     * Even when a dependency is not a direct requirement of the resource, it should be included to avoid confusion
     * surrounded the oder of events.
     *
     * @return {void}
     */
    function prepareApiKeys() {
        var when = $.when(
            app.apiKeys.isReady
        );

        when.then(setApiKeys);
    }


    /**
     * All prepare functions in this file consist of a list of dependencies for the object they are preparing
     * followed by what function telling it how to handle the success and or error.
     *
     * Even when a dependency is not a direct requirement of the resource, it should be included to avoid confusion
     * surrounded the oder of events.
     *
     * @param {jquery.Deferred} roa
     * @return {void}
     */
    function prepareRoa(roa) {
        var when = $.when(
            app.userSession.isReady,
            User.isReady
        );

        when.then(function() {
            startRoa(roa);
        });
    }


    /**
     * All prepare functions in this file consist of a list of dependencies for the object they are preparing
     * followed by what function telling it how to handle the success and or error.
     *
     * Even when a dependency is not a direct requirement of the resource, it should be included to avoid confusion
     * surrounded the oder of events.
     *
     * @param {jquery.Deferred} roa
     * @return {void}
     */
    function preparePatientIdentifiers(roa) {
        var when = $.when(
            roa,
            app.userSession.isReady,
            User.isReady
        );

        when.then(startIdentifiers);
    }


    /**
     * All prepare functions in this file consist of a list of dependencies for the object they are preparing
     * followed by what function telling it how to handle the success and or error.
     *
     * Even when a dependency is not a direct requirement of the resource, it should be included to avoid confusion
     * surrounded the oder of events.
     *
     * @param {jquery.Deferred} roa
     * @return {void}
     */
    function prepareCcTypes(roa) {
        var when = $.when(
            roa,
            app.userSession.isReady,
            User.isReady
        );

        when.then(startCcTypesOfCare);
    }

    /**
     * Called only in the event of a failure
     * @return {void}
     */
    function prepareCcTypesFail() {
        $.when(app.ccTypeOfCares.isReady).fail(onError);
    }


    /**
     * All prepare functions in this file consist of a list of dependencies for the object they are preparing
     * followed by what function telling it how to handle the success and or error.
     *
     * Even when a dependency is not a direct requirement of the resource, it should be included to avoid confusion
     * surrounded the oder of events.
     *
     * @param {jquery.Deferred} roa
     * @return {void}
     */
    function prepareFacilities(roa) {
        var when = $.when(
            roa,
            app.userSession.isReady,
            User.isReady,
            app.patientIdentifiers.isReady
        );

        when.then(startFacilities);
        when.fail(onPatientIdentifiersError);
    }

    /**
     * Called only in the event of a failure
     * @return {void}
     */
    function prepareFacilitiesFail() {
        $.when(app.facilities.isReady).fail(onError);
    }


    /**
     * All prepare functions in this file consist of a list of dependencies for the object they are preparing
     * followed by what function telling it how to handle the success and or error.
     *
     * Even when a dependency is not a direct requirement of the resource, it should be included to avoid confusion
     * surrounded the oder of events.
     *
     * @param {jquery.Deferred} roa
     * @return {void}
     */
    function prepareTracking(roa) {
        var when = $.when(
            roa,
            app.userSession.isReady,
            User.isReady,
            app.patientIdentifiers.isReady
        );

        when.then(startTracking);
    }


    /**
     * All prepare functions in this file consist of a list of dependencies for the object they are preparing
     * followed by what function telling it how to handle the success and or error.
     *
     * Even when a dependency is not a direct requirement of the resource, it should be included to avoid confusion
     * surrounded the oder of events.
     *
     * @param {jquery.Deferred} roa
     * @return {void}
     */
    function prepareRequests(roa) {
        var when = $.when(
            roa,
            app.userSession.isReady,
            User.isReady,
            app.patientIdentifiers.isReady
        );

        when.then(startRequests);
    }


    /**
     * All prepare functions in this file consist of a list of dependencies for the object they are preparing
     * followed by what function telling it how to handle the success and or error.
     *
     * Even when a dependency is not a direct requirement of the resource, it should be included to avoid confusion
     * surrounded the oder of events.
     *
     * @param {jquery.Deferred} roa
     * @return {void}
     */
    function prepareNotifications(roa) {
        var when = $.when(
            roa,
            app.userSession.isReady,
            User.isReady,
            app.patientIdentifiers.isReady
        );

        when.then(startNotfications);
    }


    /**
     * All prepare functions in this file consist of a list of dependencies for the object they are preparing
     * followed by what function telling it how to handle the success and or error.
     *
     * Even when a dependency is not a direct requirement of the resource, it should be included to avoid confusion
     * surrounded the oder of events.
     *
     * @param {jquery.Deferred} roa
     * @return {void}
     */
    function prepareNotificationPreferences(roa) {
        var when = $.when(
            roa,
            app.userSession.isReady,
            User.isReady,
            app.patientIdentifiers.isReady
        );

        when.then(startNotificationPreferences);
    }


    /**
     * All prepare functions in this file consist of a list of dependencies for the object they are preparing
     * followed by what function telling it how to handle the success and or error.
     *
     * Even when a dependency is not a direct requirement of the resource, it should be included to avoid confusion
     * surrounded the oder of events.
     *
     * @param {jquery.Deferred} roa
     * @return {void}
     */
    function preparePatientEnrolledFacilities(roa) {
        var when = $.when(
            roa,
            app.userSession.isReady,
            User.isReady,
            app.patientIdentifiers.isReady,
            app.facilities.isReady
        );

        when.then(startPatientEnrolledFacilities);
    }


    /**
     * All prepare functions in this file consist of a list of dependencies for the object they are preparing
     * followed by what function telling it how to handle the success and or error.
     *
     * Even when a dependency is not a direct requirement of the resource, it should be included to avoid confusion
     * surrounded the oder of events.
     *
     * @param {jquery.Deferred} roa
     * @return {void}
     */
    function prepareAppointments(roa) {
        var userAppDataPrepFetches = $.when(
            roa,
            app.userSession.isReady,
            User.isReady,
            app.patientIdentifiers.isReady,
            app.facilities.isReady,
            app.configurations.isReady
        );

        userAppDataPrepFetches.then(startAppointments);
    }


    /**
     * Retrieves session data
     * @return {void}
     */
    function startSession() {
        app.userSession.fetch();

    }


    /**
     * Retrieves api keys
     * @return {void}
     */
    function startApiMonitoring() {
        app.disabledFeatures.fetch();

    }


    /**
     * Retrieves list of disabled features
     * @return {void}
     */
    function startDisabledFeatures() {
        app.apiKeys.fetch();
    }


    /**
     * Checks if the user is logged in correctly and returns them landing page if not
     * @return {void}
     */
    function startUser() {
        if (app.userSession.isLoggedIn()) {
            User.fetch();
        } else {
            app.show(new LandingPageView());
            app.mainView.showChildView('footer', new FooterView());
        }
    }


    /**
     * Injects the api keys into the dom
     * @return {void}
     */
    function setApiKeys() {
        var apiKey = app.apiKeys.get('googleApiKey');
        var script = '<script src="https://maps.googleapis.com/maps/api/js?key=' + apiKey + '&libraries=distancematrix" async defer></script>';

        $('body').append(script);
    }


    /**
     * Ensures that the rights of access has been set
     * @param {jquery.Deferred} roa
     * @return {void}
     */
    function startRoa(roa) {
        if (!User.hasAcceptedROA()) {
            roaModule.navigateToRoa();
            return;
        }
        roa.resolve();
    }


    /**
     * Retrieves the patient identifiers
     * @return {void}
     */
    function startIdentifiers() {
        app.patientIdentifiers.fetch();
    }


    /**
     * Retrieves community care types of care
     * @return {void}
     */
    function startCcTypesOfCare() {
        app.ccTypeOfCares.fetch();
    }


    /**
     * Handles the error if patient identifiers fails to fetch correctly
     * @return {void}
     */
    function onPatientIdentifiersError() {
        if (!app.patientIdentifiers.hasDfnAndIcn()) {
            app.mainView.model.set('hasAppMenu', false);
            app.show(new VHARegisterView());
        } else {
            utilities.registerOauthHandler();
        }
    }


    /**
     * Beginis history and monitory
     * @return {void}
     */
    function startTracking() {
        require(['AppRouter'], function (AppRouter) {
            // AppRouter instantiates itself
            Backbone.history.start();
            app.userSession.startSessionTimer();
        });
    }


    /**
     * Retrieves or creates facility data
     * @return {void}
     */
    function startFacilities() {
        var dnfStieCodes = app.patientIdentifiers.getDfnSiteCodes();
        var parentSites;

        if (!_.isEmpty(dnfStieCodes)) {
            parentSites = dnfStieCodes.join(',');

            app.facilities.fetch({
                data: {
                    parentSiteCodes: parentSites,
                },
            });
        } else {
            app.facilities = new Backbone.Collection({});
        }
    }


    /**
     * Retrieves the list of the users requested appointments
     * @return {void}
     */
    function startRequests() {
        var userURIComponents = getUriComponents();
        if (isAppointmentsReadyToLoad(userURIComponents)) {
            appointmentsModule.requests.fetch({
                url: app.resources.get('appointment-requests').formatHref(userURIComponents),
            });
        }
    }


    /**
     * Retrieves the users notification preferences
     * @return {void}
     */
    function startNotificationPreferences() {
        var userURIComponents = getUriComponents();
        if (isAppointmentsReadyToLoad(userURIComponents)) {
            notificationPreferencesModule.notificationPreferences.fetch({
                url: app.resources.get('notification-preference').formatHref(userURIComponents),
            });
        }
    }


    /**
     * Retrieves a list of notification
     * @return {void}
     */
    function startNotfications() {
        var userURIComponents = getUriComponents();
        if (isAppointmentsReadyToLoad(userURIComponents)) {
            notificationsModule.fetchCollection(userURIComponents);
        }
    }


    /**
     * Finds the facilities with the sites supporting var flag
     * @return {void}
     */
    function startPatientEnrolledFacilities() {
        var enrolledSiteCodes = app.patientIdentifiers.getDfnSiteCodes();
        var ajaxFilter = function () {
            return $.ajax({
                url: app.resources.get('supported-facilities').get('href'),
                data: { siteCodes: this.pluck('facilityCode').join() },
                method: 'GET',
                dataType: 'json',
                cache: false,
                success: function (response) {
                    if (!_.isUndefined(response.sitesSupportingVAR)) {
                        _.each(response.sitesSupportingVAR, function (facility) {
                            this.get(facility.id).set('supportsVAR', true);
                        }.bind(this));
                    }
                }.bind(this),
            });
        }.bind(app.patientEnrolledFacilities);

        app.patientEnrolledFacilities.setWith(enrolledSiteCodes, app.facilities, ajaxFilter);
    }


    /**
     * Retrieves a loist of booked appointments
     * @return {void}
     */
    function startAppointments() {
        var userURIComponents = getUriComponents();
        if (typeof app.configurations.numberOfMonthsForFutureAppointments !== 'undefined') {
            appointmentsModule.appointments.fetchAppointmentData(
                app.patientEnrolledFacilities,
                app.resources.get('ds-booked-appointments'),
                app.resources.get('ds-booked-cc-appointments'),
                userURIComponents
            );
        }
    }


    /**
     * @param {object} userURIComponents
     * @return {boolean}
     */
    function isAppointmentsReadyToLoad(userURIComponents) {
        return userURIComponents['assigning-authority']
            && userURIComponents['patient-id']
            && app.patientIdentifiers.hasDfnAndIcn();
    }


    /**
     * @return {{'assigning-authority': string, 'patient-id': string}}
     */
    function getUriComponents() {
        return {
            'assigning-authority': User.get('idType'),
            'patient-id': User.get('id'),
        };
    }


    /**
     * Retrieves configuration from the server
     * @return {void}
     */
    function getConfigurations() {
        app.configurations.fetch();
    }

    /**
     * Get CC Eligibility Info from ADR service
     * @return {void}
     */
    function prepareCCEligibilityDetermination() {
        var when = $.when(
            app.userSession.isReady,
            User.isReady,
            app.ccEligibilityModel.isReady
        );
        when.then(startCCEligibilityDetermination);
    }

    function startCCEligibilityDetermination() {
        if (app.isCcEligFeatureEnabled()) {
            app.ccEligibilityModel.fetch();
        }
    }

    app.start();
});
